from flask import request
from flask_jwt_extended import jwt_required

from app.http.requests.v1.notification_request import (
    UpdateNotificationReadRequest,
    GetNotificationRequest,
    GetBadgeRequest,
    UpdateNotificationAllReadRequest,
)
from app.http.responses.presenters.v1.notification_presenter import (
    UpdateNotificationReadPresenter,
    GetNotificationPresenter,
    GetBadgePresenter,
)
from app.http.view import api, auth_required, current_user
from core.domains.notification.use_case.v1.notification_use_case import (
    UpdateNotificationReadUseCase,
    GetNotificationUseCase,
    GetBadgeUseCase,
    UpdateNotificationAllReadUseCase,
)


@api.route("/v1/notifications", methods=["GET"])
@jwt_required
@auth_required
def get_notification_view():
    dto = GetNotificationRequest(
        previous_id=request.args.get("previous_id"), user_id=current_user.id
    ).validate_request_and_make_dto()
    return GetNotificationPresenter().transform(
        GetNotificationUseCase().execute(dto=dto)
    )


@api.route("/v1/notifications/badge", methods=["GET"])
@jwt_required
@auth_required
def get_badge_view():
    dto = GetBadgeRequest(user_id=current_user.id).validate_request_and_make_dto()
    return GetBadgePresenter().transform(GetBadgeUseCase().execute(dto=dto))


@api.route("/v1/notifications/<int:notification_id>/read", methods=["PATCH"])
@jwt_required
def update_notification_read_view(notification_id):
    dto = UpdateNotificationReadRequest(
        notification_id=notification_id,
    ).validate_request_and_make_dto()

    return UpdateNotificationReadPresenter().transform(
        UpdateNotificationReadUseCase().execute(dto=dto)
    )


@api.route("/v1/notifications/read", methods=["PATCH"])
@jwt_required
@auth_required
def update_notification_all_read_view():
    dto = UpdateNotificationAllReadRequest(
        user_id=current_user.id
    ).validate_request_and_make_dto()

    return UpdateNotificationReadPresenter().transform(
        UpdateNotificationAllReadUseCase().execute(dto=dto)
    )
