import io

import pytest
from faker.generator import random
from werkzeug.datastructures import FileStorage

from app.http.requests.v1.post_request import CreatePostRequest, UpdatePostRequest
from core.exceptions import InvalidRequestException
from tests.seeder.conftest import faker


def test_create_post_image_upload_request_should_limit_format(create_user):
    test_image_1 = FileStorage(
        stream=io.BytesIO(b"aaa"),
        filename=f"test.png",
        content_type="multipart/form-data",
    )

    test_image_2 = FileStorage(
        stream=io.BytesIO(b"aaa"),
        filename=f"test.jpg",
        content_type="multipart/form-data",
    )

    test_image_3 = FileStorage(
        stream=io.BytesIO(b"aaa"),
        filename=f"test.jpeg",
        content_type="multipart/form-data",
    )
    files: list = [test_image_1, test_image_2, test_image_3]

    result = CreatePostRequest(
        user_id=create_user.id,
        post_category_id=random.choice([1, 2, 3, 4]),
        body=faker.sentence(),
        files=files,
    ).validate_request_and_make_dto()

    for result_file, test_file in zip(result.files, files):
        assert result_file == test_file


def test_create_post_image_upload_request_should_fail_when_wrong_format(create_user):
    test_image_1 = FileStorage(
        stream=io.BytesIO(b"aaa"),
        filename=f"test.txt",
        content_type="multipart/form-data",
    )

    test_image_2 = FileStorage(
        stream=io.BytesIO(b"aaa"),
        filename=f"test.png",
        content_type="multipart/form-data",
    )

    test_image_3 = FileStorage(
        stream=io.BytesIO(b"aaa"),
        filename=f"test.jpg",
        content_type="multipart/form-data",
    )
    files: list = [test_image_1, test_image_2, test_image_3]

    with pytest.raises(InvalidRequestException):
        CreatePostRequest(
            user_id=create_user.id,
            post_category_id=random.choice([1, 2, 3, 4]),
            body=faker.sentence(),
            files=files,
        ).validate_request_and_make_dto()


def test_update_post_image_request_should_limit_format(create_user, create_post):
    test_image_1 = FileStorage(
        stream=io.BytesIO(b"aaa"),
        filename=f"test.png",
        content_type="multipart/form-data",
    )

    test_image_2 = FileStorage(
        stream=io.BytesIO(b"aaa"),
        filename=f"test.jpg",
        content_type="multipart/form-data",
    )

    test_image_3 = FileStorage(
        stream=io.BytesIO(b"aaa"),
        filename=f"test.jpeg",
        content_type="multipart/form-data",
    )

    files: list = [test_image_1, test_image_2, test_image_3]

    result = UpdatePostRequest(
        post_id=create_post.id,
        user_id=create_user.id,
        post_category_id=random.choice([1, 2, 3, 4]),
        body=faker.sentence(),
        files=files,
        delete_files=[],
    ).validate_request_and_make_dto()

    for result_file, test_file in zip(result.files, files):
        assert result_file == test_file


def test_update_post_image_upload_request_should_fail_when_wrong_format(
    create_user, create_post
):
    test_image_1 = FileStorage(
        stream=io.BytesIO(b"aaa"),
        filename=f"test.txt",
        content_type="multipart/form-data",
    )

    test_image_2 = FileStorage(
        stream=io.BytesIO(b"aaa"),
        filename=f"test.png",
        content_type="multipart/form-data",
    )

    test_image_3 = FileStorage(
        stream=io.BytesIO(b"aaa"),
        filename=f"test.jpg",
        content_type="multipart/form-data",
    )

    files: list = [test_image_1, test_image_2, test_image_3]

    with pytest.raises(InvalidRequestException):
        UpdatePostRequest(
            post_id=create_post.id,
            user_id=create_user.id,
            post_category_id=random.choice([1, 2, 3, 4]),
            body=faker.sentence(),
            files=files,
            delete_files=[],
        ).validate_request_and_make_dto()
