import io
import json
import os
import random
import uuid
from unittest.mock import patch

from faker import Faker
from flask import url_for
from werkzeug.datastructures import FileStorage

from app.extensions.utils.enum.aws_enum import S3PathEnum, CloudFrontEnum
from app.persistence.model import PostModel
from core.domains.community.dto.community_dto import CreatePostAttachmentDto

faker = Faker()


@patch(
    "core.domains.cs.use_case.v1.cs_use_case.CreateCSUseCase._upload_inquiry_attachments",
    return_value=None,
)
def test_view_create_cs_when_with_files_then_success(
    _upload_inquiry_attachments,
    client,
    session,
    test_request_context,
    make_header,
    make_authorization,
    create_user,
):
    authorization = make_authorization(user_id=create_user.id)
    headers = make_header(
        authorization=authorization, content_type="multipart/form-data", accept="*/*"
    )

    # 실제 업로드 확인하려면 아래 경로에 이미지 첨부하고 patch 데코레이터 제거한 뒤 실행.
    file1 = FileStorage(
        stream=io.BytesIO(b"aaa"),
        filename="/Users/noah/Downloads/aa.png",
        content_type="multipart/form-data",
    )
    file2 = FileStorage(
        stream=io.BytesIO(b"aaa"),
        filename="/Users/noah/Downloads/bb.jpeg",
        content_type="multipart/form-data",
    )

    param = dict(
        title=faker.sentence(),
        body=faker.sentence(),
        email=faker.email(),
        files=[file1, file2],
    )

    with test_request_context:
        response = client.post(
            url_for("api/groot.create_cs"), headers=headers, data=param
        )

    result = response.get_json()["data"]["result"]
    assert response.status_code == 200
    assert result == "success"


def test_view_create_cs_when_without_files_then_success(
    client, session, test_request_context, make_header, make_authorization, create_user,
):
    authorization = make_authorization(user_id=create_user.id)
    headers = make_header(
        authorization=authorization, content_type="multipart/form-data", accept="*/*"
    )

    param = dict(
        title=faker.sentence(), body=faker.sentence(), email=faker.email(), files=[],
    )

    with test_request_context:
        response = client.post(
            url_for("api/groot.create_cs"), headers=headers, data=param
        )

    result = response.get_json()["data"]["result"]
    assert response.status_code == 200
    assert result == "success"


def test_view_create_cs_when_over_5_picture_then_failure(
    client, session, test_request_context, make_header, make_authorization, create_user,
):
    authorization = make_authorization(user_id=create_user.id)
    headers = make_header(
        authorization=authorization, content_type="multipart/form-data", accept="*/*"
    )

    files = list()
    for _ in range(0, 6):
        file = FileStorage(
            stream=io.BytesIO(b"aaa"),
            filename="/Users/noah/Downloads/aa.png",
            content_type="multipart/form-data",
        )
        files.append(file)

    param = dict(
        title=faker.sentence(), body=faker.sentence(), email=faker.email(), files=files,
    )

    with test_request_context:
        response = client.post(
            url_for("api/groot.create_cs"), headers=headers, data=param
        )

    result = response.get_json()
    assert response.status_code == 400
    assert result.get("detail") == "over 5 picture"
    assert result.get("message") == "invalid_request_error"


def test_view_create_cs_when_empty_body_then_failure(
    client, session, test_request_context, make_header, make_authorization, create_user,
):
    authorization = make_authorization(user_id=create_user.id)
    headers = make_header(
        authorization=authorization, content_type="multipart/form-data", accept="*/*"
    )

    param = dict(title=faker.sentence(), email=faker.email(), body="", files=[],)

    with test_request_context:
        response = client.post(
            url_for("api/groot.create_cs"), headers=headers, data=param
        )

    result = response.get_json()
    assert response.status_code == 400
    assert result.get("detail") == "empty body"
    assert result.get("message") == "invalid_request_error"


def test_view_create_cs_when_over_char_body_then_failure(
    client, session, test_request_context, make_header, make_authorization, create_user,
):
    authorization = make_authorization(user_id=create_user.id)
    headers = make_header(
        authorization=authorization, content_type="multipart/form-data", accept="*/*"
    )

    param = dict(
        title=faker.sentence(), email=faker.email(), body="a" * 1001, files=[],
    )

    with test_request_context:
        response = client.post(
            url_for("api/groot.create_cs"), headers=headers, data=param
        )

    result = response.get_json()
    assert response.status_code == 400
    assert result.get("detail") == "over 1000 char"
    assert result.get("message") == "invalid_request_error"
