from pydantic import BaseModel, ValidationError, StrictInt

from app.extensions.utils.log_helper import logger_
from core.domains.notification.dto.notification_dto import (
    UpdateNotificationReadDto,
    GetNotificationDto,
    GetBadgeDto,
    UpdateNotificationAllReadDto,
)
from core.exceptions import InvalidRequestException

logger = logger_.getLogger(__name__)


class UpdateNotificationReadSchema(BaseModel):
    notification_id: StrictInt


class UpdateNotificationAllReadSchema(BaseModel):
    user_id: StrictInt


class GetNotificationSchema(BaseModel):
    user_id: StrictInt
    previous_id: StrictInt | None


class GetBadgeSchema(BaseModel):
    user_id: StrictInt


class UpdateNotificationReadRequest:
    def __init__(self, notification_id):
        self.notification_id = int(notification_id) if notification_id else None

    def validate_request_and_make_dto(self):
        try:
            schema = UpdateNotificationReadSchema(
                notification_id=self.notification_id,
            ).dict()
            return UpdateNotificationReadDto(**schema)
        except ValidationError as e:
            logger.error(
                f"[UpdateNotificationReadRequest][validate_request_and_make_dto] error : {e}"
            )
            raise InvalidRequestException(message=e.errors())


class UpdateNotificationAllReadRequest:
    def __init__(self, user_id):
        self.user_id = int(user_id) if user_id else None

    def validate_request_and_make_dto(self):
        try:
            schema = UpdateNotificationAllReadSchema(user_id=self.user_id,).dict()
            return UpdateNotificationAllReadDto(**schema)
        except ValidationError as e:
            logger.error(
                f"[UpdateNotificationAllReadRequest][validate_request_and_make_dto] error : {e}"
            )
            raise InvalidRequestException(message=e.errors())


class GetNotificationRequest:
    def __init__(self, user_id, previous_id):
        self.user_id = int(user_id) if user_id else None
        self.previous_id = int(previous_id) if previous_id else None

    def validate_request_and_make_dto(self):
        try:
            schema = GetNotificationSchema(
                user_id=self.user_id, previous_id=self.previous_id
            ).dict()
            return GetNotificationDto(**schema)
        except ValidationError as e:
            logger.error(
                f"[GetNotificationRequest][validate_request_and_make_dto] error : {e}"
            )
            raise InvalidRequestException(message=e.errors())


class GetBadgeRequest:
    def __init__(self, user_id):
        self.user_id = int(user_id) if user_id else None

    def validate_request_and_make_dto(self):
        try:
            schema = GetBadgeSchema(user_id=self.user_id,).dict()
            return GetBadgeDto(**schema)
        except ValidationError as e:
            logger.error(
                f"[GetBadgeRequest][validate_request_and_make_dto] error : {e}"
            )
            raise InvalidRequestException(message=e.errors())
