from http import HTTPStatus

from pydantic import ValidationError

from app.extensions.utils.log_helper import logger_
from app.http.responses import failure_response, success_response
from core.domains.authentication.schema.authentication import (
    AuthenticationResponseSchema,
    LogoutResponseSchema,
)
from core.domains.oauth.schema.oauth_schema import VerificationResponseSchema
from core.use_case_output import UseCaseSuccessOutput, FailureType, UseCaseFailureOutput


logger = logger_.getLogger(__name__)


class AuthenticationPresenter:
    def transform(
        self, response: UseCaseSuccessOutput,
    ):
        try:
            schema = AuthenticationResponseSchema(result=response.type)
        except ValidationError as e:
            return failure_response(
                UseCaseFailureOutput(
                    type=FailureType.SYSTEM_ERROR,
                    message="response schema validation error",
                    code=HTTPStatus.INTERNAL_SERVER_ERROR,
                )
            )
        result = {
            "data": schema.dict(),
            "meta": response.meta,
        }
        return success_response(result=result)


class LogoutPresenter:
    def transform(self, output: UseCaseSuccessOutput | UseCaseFailureOutput):
        if isinstance(output, UseCaseSuccessOutput):
            value = output.value
            value_to_json = value.get_json()
            try:
                schema = LogoutResponseSchema(
                    blacklist_token=value_to_json.get("blacklist_token"),
                    expired_at=value_to_json.get("expired_at"),
                )
            except ValidationError as e:
                logger.error(
                    f"[UpdateJwtPresenter][transform] value : {value} error : {e}"
                )
                return failure_response(
                    UseCaseFailureOutput(
                        message=FailureType.INTERNAL_ERROR,
                        type="response schema validation error",
                    ),
                    status_code=HTTPStatus.INTERNAL_SERVER_ERROR,
                )
            result = {
                "data": schema.dict(),
            }
            return success_response(result=result)
        elif isinstance(output, UseCaseFailureOutput):
            return failure_response(output=output)


class VerificationJWTPresenter:
    def transform(self, output: UseCaseSuccessOutput | UseCaseFailureOutput):
        if isinstance(output, UseCaseSuccessOutput):
            value = output.value
            value_to_json = value.get_json()
            try:
                schema = VerificationResponseSchema(
                    access_token=value_to_json.get("access_token")
                )
            except ValidationError as e:
                logger.error(
                    f"[VerificationPresenter][transform] value : {value} error : {e}"
                )
                return failure_response(
                    UseCaseFailureOutput(
                        message=FailureType.INTERNAL_ERROR,
                        type="response schema validation error",
                    )
                )
            result = {
                "data": schema.dict(),
            }
            return success_response(result=result)
        elif isinstance(output, UseCaseFailureOutput):
            return failure_response(output=output)
