from http import HTTPStatus

from pydantic import ValidationError

from app.http.responses import failure_response, success_response
from core.domains.user.schema.user_schema import (
    UploadUserProfileImgResponseSchema,
    DefaultUserResponseSchema,
    GetUserResponseSchema,
    GetReceivePushTypeResponseSchema,
    CreateUserResponseSchema,
    GetMacResponseSchema,
)
from core.use_case_output import UseCaseSuccessOutput, UseCaseFailureOutput, FailureType


class UploadUserProfileImgPresenter:
    def transform(self, output: UseCaseSuccessOutput | UseCaseFailureOutput):
        if isinstance(output, UseCaseSuccessOutput):
            try:
                result = output.value.dict()
                schema = UploadUserProfileImgResponseSchema(**result)
            except ValidationError:
                return failure_response(
                    UseCaseFailureOutput(
                        type="response schema validation error",
                        message=FailureType.INTERNAL_ERROR,
                        code=HTTPStatus.INTERNAL_SERVER_ERROR,
                    ),
                    status_code=HTTPStatus.INTERNAL_SERVER_ERROR,
                )
            result = {
                "data": schema.dict(),
                "meta": output.meta,
            }
            return success_response(result=result)
        elif isinstance(output, UseCaseFailureOutput):
            return failure_response(output=output, status_code=output.code)


class CreateUserPresenter:
    def transform(self, output: UseCaseSuccessOutput | UseCaseFailureOutput):
        if isinstance(output, UseCaseSuccessOutput):
            value = output.value
            value_to_json = value.get_json()
            try:
                schema = CreateUserResponseSchema(
                    access_token=value_to_json.get("access_token")
                )
            except ValidationError:
                return failure_response(
                    UseCaseFailureOutput(
                        type="response schema validation error",
                        message=FailureType.INTERNAL_ERROR,
                        code=HTTPStatus.INTERNAL_SERVER_ERROR,
                    ),
                    status_code=HTTPStatus.INTERNAL_SERVER_ERROR,
                )
            result = {
                "data": schema.dict(),
            }
            return success_response(result=result)
        elif isinstance(output, UseCaseFailureOutput):
            return failure_response(output=output, status_code=output.code)


class GetUserPresenter:
    def transform(self, output: UseCaseSuccessOutput | UseCaseFailureOutput):
        if isinstance(output, UseCaseSuccessOutput):
            try:
                schema = GetUserResponseSchema(
                    user=output.value.get("user"),
                    plant_categories=output.value.get("plant_categories"),
                )
            except ValidationError:
                return failure_response(
                    UseCaseFailureOutput(
                        type="response schema validation error",
                        message=FailureType.INTERNAL_ERROR,
                        code=HTTPStatus.INTERNAL_SERVER_ERROR,
                    ),
                    status_code=HTTPStatus.INTERNAL_SERVER_ERROR,
                )
            result = {
                "data": schema.dict(),
                "meta": output.meta,
            }
            return success_response(result=result)
        elif isinstance(output, UseCaseFailureOutput):
            return failure_response(output=output, status_code=output.code)


class UpdateUserPresenter:
    def transform(self, output: UseCaseSuccessOutput | UseCaseFailureOutput):
        if isinstance(output, UseCaseSuccessOutput):
            try:
                schema = DefaultUserResponseSchema(result=output.type)
            except ValidationError:
                return failure_response(
                    UseCaseFailureOutput(
                        type="response schema validation error",
                        message=FailureType.INTERNAL_ERROR,
                        code=HTTPStatus.INTERNAL_SERVER_ERROR,
                    ),
                    status_code=HTTPStatus.INTERNAL_SERVER_ERROR,
                )
            result = {
                "data": schema.dict(),
                "meta": output.meta,
            }
            return success_response(result=result)
        elif isinstance(output, UseCaseFailureOutput):
            return failure_response(output=output, status_code=output.code)


class UpdateReceivePushTypePresenter:
    def transform(self, output: UseCaseSuccessOutput | UseCaseFailureOutput):
        if isinstance(output, UseCaseSuccessOutput):
            try:
                schema = DefaultUserResponseSchema(result=output.type)
            except ValidationError:
                return failure_response(
                    UseCaseFailureOutput(
                        type="response schema validation error",
                        message=FailureType.INTERNAL_ERROR,
                        code=HTTPStatus.INTERNAL_SERVER_ERROR,
                    ),
                    status_code=HTTPStatus.INTERNAL_SERVER_ERROR,
                )
            result = {
                "data": schema.dict(),
                "meta": output.meta,
            }
            return success_response(result=result)
        elif isinstance(output, UseCaseFailureOutput):
            return failure_response(output=output, status_code=output.code)


class GetReceivePushTypePresenter:
    def transform(self, output: UseCaseSuccessOutput | UseCaseFailureOutput):
        if isinstance(output, UseCaseSuccessOutput):
            try:
                schema = GetReceivePushTypeResponseSchema(
                    receive_push_type=output.value
                )
            except ValidationError:
                return failure_response(
                    UseCaseFailureOutput(
                        type="response schema validation error",
                        message=FailureType.INTERNAL_ERROR,
                        code=HTTPStatus.INTERNAL_SERVER_ERROR,
                    ),
                    status_code=HTTPStatus.INTERNAL_SERVER_ERROR,
                )
            result = {
                "data": schema.dict(),
                "meta": output.meta,
            }
            return success_response(result=result)
        elif isinstance(output, UseCaseFailureOutput):
            return failure_response(output=output, status_code=output.code)


class UpdateDevicePresenter:
    def transform(self, output: UseCaseSuccessOutput | UseCaseFailureOutput):
        if isinstance(output, UseCaseSuccessOutput):
            try:
                schema = DefaultUserResponseSchema(result=output.type)
            except ValidationError:
                return failure_response(
                    UseCaseFailureOutput(
                        type="response schema validation error",
                        message=FailureType.INTERNAL_ERROR,
                        code=HTTPStatus.INTERNAL_SERVER_ERROR,
                    ),
                    status_code=HTTPStatus.INTERNAL_SERVER_ERROR,
                )
            result = {
                "data": schema.dict(),
                "meta": output.meta,
            }
            return success_response(result=result)
        elif isinstance(output, UseCaseFailureOutput):
            return failure_response(output=output, status_code=output.code)


class UpdateFCMTokenPresenter:
    def transform(self, output: UseCaseSuccessOutput | UseCaseFailureOutput):
        if isinstance(output, UseCaseSuccessOutput):
            try:
                schema = DefaultUserResponseSchema(result=output.type)
            except ValidationError:
                return failure_response(
                    UseCaseFailureOutput(
                        type="response schema validation error",
                        message=FailureType.INTERNAL_ERROR,
                        code=HTTPStatus.INTERNAL_SERVER_ERROR,
                    ),
                    status_code=HTTPStatus.INTERNAL_SERVER_ERROR,
                )
            result = {
                "data": schema.dict(),
                "meta": output.meta,
            }
            return success_response(result=result)
        elif isinstance(output, UseCaseFailureOutput):
            return failure_response(output=output, status_code=output.code)


class GetMacPresenter:
    def transform(self, output: UseCaseSuccessOutput | UseCaseFailureOutput):
        if isinstance(output, UseCaseSuccessOutput):
            try:
                schema = GetMacResponseSchema(hw_mac=output.value)
            except ValidationError:
                return failure_response(
                    UseCaseFailureOutput(
                        type="response schema validation error",
                        message=FailureType.INTERNAL_ERROR,
                        code=HTTPStatus.INTERNAL_SERVER_ERROR,
                    ),
                    status_code=HTTPStatus.INTERNAL_SERVER_ERROR,
                )
            result = {
                "data": schema.dict(),
                "meta": output.meta,
            }
            return success_response(result=result)
        elif isinstance(output, UseCaseFailureOutput):
            return failure_response(output=output, status_code=output.code)
