from http import HTTPStatus

from pydantic import ValidationError

from app.http.responses import failure_response, success_response
from core.domains.community.schema.community_schema import (
    GetPostCategoryResponseSchema,
    DefaultPostResponseSchema,
    GetPostResponseSchema,
    GetFeedsResponseSchema,
    CreateCommentResponseSchema,
    GetNoticesResponseSchema,
)
from core.use_case_output import UseCaseSuccessOutput, UseCaseFailureOutput, FailureType


class GetPostCategoryPresenter:
    def transform(self, output: UseCaseSuccessOutput | UseCaseFailureOutput):
        if isinstance(output, UseCaseSuccessOutput):
            try:
                schema = GetPostCategoryResponseSchema(categories=output.value)
            except ValidationError:
                return failure_response(
                    UseCaseFailureOutput(
                        type="response schema validation error",
                        message=FailureType.INTERNAL_ERROR,
                        code=HTTPStatus.INTERNAL_SERVER_ERROR,
                    ),
                    status_code=HTTPStatus.INTERNAL_SERVER_ERROR,
                )
            result = {
                "data": schema.dict(),
                "meta": output.meta,
            }
            return success_response(result=result)
        elif isinstance(output, UseCaseFailureOutput):
            return failure_response(output=output, status_code=output.code)


class CreatePostPresenter:
    def transform(self, output: UseCaseSuccessOutput | UseCaseFailureOutput):
        if isinstance(output, UseCaseSuccessOutput):
            try:
                schema = DefaultPostResponseSchema(result=output.type)
            except ValidationError:
                return failure_response(
                    UseCaseFailureOutput(
                        type="response schema validation error",
                        message=FailureType.INTERNAL_ERROR,
                        code=HTTPStatus.INTERNAL_SERVER_ERROR,
                    ),
                    status_code=HTTPStatus.INTERNAL_SERVER_ERROR,
                )
            result = {
                "data": schema.dict(),
                "meta": output.meta,
            }
            return success_response(result=result)
        elif isinstance(output, UseCaseFailureOutput):
            return failure_response(output=output, status_code=output.code)


class UpdatePostPresenter:
    def transform(self, output: UseCaseSuccessOutput | UseCaseFailureOutput):
        if isinstance(output, UseCaseSuccessOutput):
            try:
                schema = DefaultPostResponseSchema(result=output.type)
            except ValidationError:
                return failure_response(
                    UseCaseFailureOutput(
                        type="response schema validation error",
                        message=FailureType.INTERNAL_ERROR,
                        code=HTTPStatus.INTERNAL_SERVER_ERROR,
                    ),
                    status_code=HTTPStatus.INTERNAL_SERVER_ERROR,
                )
            result = {
                "data": schema.dict(),
                "meta": output.meta,
            }
            return success_response(result=result)
        elif isinstance(output, UseCaseFailureOutput):
            return failure_response(output=output, status_code=output.code)


class DeletePostPresenter:
    def transform(self, output: UseCaseSuccessOutput | UseCaseFailureOutput):
        if isinstance(output, UseCaseSuccessOutput):
            try:
                schema = DefaultPostResponseSchema(result=output.type)
            except ValidationError:
                return failure_response(
                    UseCaseFailureOutput(
                        type="response schema validation error",
                        message=FailureType.INTERNAL_ERROR,
                        code=HTTPStatus.INTERNAL_SERVER_ERROR,
                    ),
                    status_code=HTTPStatus.INTERNAL_SERVER_ERROR,
                )
            result = {
                "data": schema.dict(),
                "meta": output.meta,
            }
            return success_response(result=result)
        elif isinstance(output, UseCaseFailureOutput):
            return failure_response(output=output, status_code=output.code)


class GetPostPresenter:
    def transform(self, output: UseCaseSuccessOutput | UseCaseFailureOutput):
        if isinstance(output, UseCaseSuccessOutput):
            try:
                schema = GetPostResponseSchema(
                    post=output.value.get("post"),
                    post_categories=output.value.get("post_categories"),
                )
            except ValidationError:
                return failure_response(
                    UseCaseFailureOutput(
                        type="response schema validation error",
                        message=FailureType.INTERNAL_ERROR,
                        code=HTTPStatus.INTERNAL_SERVER_ERROR,
                    ),
                    status_code=HTTPStatus.INTERNAL_SERVER_ERROR,
                )
            result = {
                "data": schema.dict(),
                "meta": output.meta,
            }
            return success_response(result=result)
        elif isinstance(output, UseCaseFailureOutput):
            return failure_response(output=output, status_code=output.code)


class UpdatePostLikeStatusPresenter:
    def transform(self, output: UseCaseSuccessOutput | UseCaseFailureOutput):
        if isinstance(output, UseCaseSuccessOutput):
            try:
                schema = DefaultPostResponseSchema(result=output.type)
            except ValidationError:
                return failure_response(
                    UseCaseFailureOutput(
                        type="response schema validation error",
                        message=FailureType.INTERNAL_ERROR,
                        code=HTTPStatus.INTERNAL_SERVER_ERROR,
                    ),
                    status_code=HTTPStatus.INTERNAL_SERVER_ERROR,
                )
            result = {
                "data": schema.dict(),
                "meta": output.meta,
            }
            return success_response(result=result)
        elif isinstance(output, UseCaseFailureOutput):
            return failure_response(output=output, status_code=output.code)


class CreateCommentPresenter:
    def transform(self, output: UseCaseSuccessOutput | UseCaseFailureOutput):
        if isinstance(output, UseCaseSuccessOutput):
            try:
                schema = CreateCommentResponseSchema(comments=output.value)
            except ValidationError:
                return failure_response(
                    UseCaseFailureOutput(
                        type="response schema validation error",
                        message=FailureType.INTERNAL_ERROR,
                        code=HTTPStatus.INTERNAL_SERVER_ERROR,
                    ),
                    status_code=HTTPStatus.INTERNAL_SERVER_ERROR,
                )
            result = {
                "data": schema.dict(),
                "meta": output.meta,
            }
            return success_response(result=result)
        elif isinstance(output, UseCaseFailureOutput):
            return failure_response(output=output, status_code=output.code)


class UpdateCommentPresenter:
    def transform(self, output: UseCaseSuccessOutput | UseCaseFailureOutput):
        if isinstance(output, UseCaseSuccessOutput):
            try:
                schema = DefaultPostResponseSchema(result=output.type)
            except ValidationError:
                return failure_response(
                    UseCaseFailureOutput(
                        type="response schema validation error",
                        message=FailureType.INTERNAL_ERROR,
                        code=HTTPStatus.INTERNAL_SERVER_ERROR,
                    ),
                    status_code=HTTPStatus.INTERNAL_SERVER_ERROR,
                )
            result = {
                "data": schema.dict(),
                "meta": output.meta,
            }
            return success_response(result=result)
        elif isinstance(output, UseCaseFailureOutput):
            return failure_response(output=output, status_code=output.code)


class DeleteCommentPresenter:
    def transform(self, output: UseCaseSuccessOutput | UseCaseFailureOutput):
        if isinstance(output, UseCaseSuccessOutput):
            try:
                schema = DefaultPostResponseSchema(result=output.type)
            except ValidationError:
                return failure_response(
                    UseCaseFailureOutput(
                        type="response schema validation error",
                        message=FailureType.INTERNAL_ERROR,
                        code=HTTPStatus.INTERNAL_SERVER_ERROR,
                    ),
                    status_code=HTTPStatus.INTERNAL_SERVER_ERROR,
                )
            result = {
                "data": schema.dict(),
                "meta": output.meta,
            }
            return success_response(result=result)
        elif isinstance(output, UseCaseFailureOutput):
            return failure_response(output=output, status_code=output.code)


class UpdateCommentLikeStatusPresenter:
    def transform(self, output: UseCaseSuccessOutput | UseCaseFailureOutput):
        if isinstance(output, UseCaseSuccessOutput):
            try:
                schema = DefaultPostResponseSchema(result=output.type)
            except ValidationError:
                return failure_response(
                    UseCaseFailureOutput(
                        type="response schema validation error",
                        message=FailureType.INTERNAL_ERROR,
                        code=HTTPStatus.INTERNAL_SERVER_ERROR,
                    ),
                    status_code=HTTPStatus.INTERNAL_SERVER_ERROR,
                )
            result = {
                "data": schema.dict(),
                "meta": output.meta,
            }
            return success_response(result=result)
        elif isinstance(output, UseCaseFailureOutput):
            return failure_response(output=output, status_code=output.code)


class GetFeedsPresenter:
    def transform(self, output: UseCaseSuccessOutput | UseCaseFailureOutput):
        if isinstance(output, UseCaseSuccessOutput):
            try:
                schema = GetFeedsResponseSchema(
                    feeds=output.value.get("data"), cursor=output.value.get("cursor")
                )
            except ValidationError:
                return failure_response(
                    UseCaseFailureOutput(
                        type="response schema validation error",
                        message=FailureType.INTERNAL_ERROR,
                        code=HTTPStatus.INTERNAL_SERVER_ERROR,
                    ),
                    status_code=HTTPStatus.INTERNAL_SERVER_ERROR,
                )
            result = {
                "data": schema.dict(),
                "meta": output.meta,
            }
            return success_response(result=result)
        elif isinstance(output, UseCaseFailureOutput):
            return failure_response(output=output, status_code=output.code)


class GetNoticesPresenter:
    def transform(self, output: UseCaseSuccessOutput | UseCaseFailureOutput):
        if isinstance(output, UseCaseSuccessOutput):
            try:
                schema = GetNoticesResponseSchema(
                    notices=output.value.get("data"), cursor=output.value.get("cursor")
                )
            except ValidationError:
                return failure_response(
                    UseCaseFailureOutput(
                        type="response schema validation error",
                        message=FailureType.INTERNAL_ERROR,
                        code=HTTPStatus.INTERNAL_SERVER_ERROR,
                    ),
                    status_code=HTTPStatus.INTERNAL_SERVER_ERROR,
                )
            result = {
                "data": schema.dict(),
                "meta": output.meta,
            }
            return success_response(result=result)
        elif isinstance(output, UseCaseFailureOutput):
            return failure_response(output=output, status_code=output.code)


class GetPushFeedPresenter:
    def transform(self, output: UseCaseSuccessOutput | UseCaseFailureOutput):
        if isinstance(output, UseCaseSuccessOutput):
            try:
                schema = GetFeedsResponseSchema(
                    feeds=output.value.get("data"), cursor=output.value.get("cursor")
                )
            except ValidationError:
                return failure_response(
                    UseCaseFailureOutput(
                        type="response schema validation error",
                        message=FailureType.INTERNAL_ERROR,
                        code=HTTPStatus.INTERNAL_SERVER_ERROR,
                    ),
                    status_code=HTTPStatus.INTERNAL_SERVER_ERROR,
                )
            result = {
                "data": schema.dict(),
                "meta": output.meta,
            }
            return success_response(result=result)
        elif isinstance(output, UseCaseFailureOutput):
            return failure_response(output=output, status_code=output.code)
