import io
from unittest.mock import patch

import pytest
from werkzeug.datastructures import FileStorage

from app.extensions.utils.enum.image_enum import ImageSizeEnum
from app.http.requests.v1.user_request import UploadUserProfileImgRequestSchema
from core.exceptions import InvalidRequestException


def test_image_upload_request_should_limit_format():
    file = FileStorage(
        stream=io.BytesIO(b"aaa"),
        filename=f"test.png",
        content_type="multipart/form-data",
    )

    result = UploadUserProfileImgRequestSchema(
        file=file
    ).validate_request_and_make_dto()

    assert result.file == file


def test_image_upload_request_should_fail_when_wrong_format():
    file = FileStorage(
        stream=io.BytesIO(b"aaa"),
        filename=f"test.txt",
        content_type="multipart/form-data",
    )

    with pytest.raises(InvalidRequestException):
        UploadUserProfileImgRequestSchema(file=file).validate_request_and_make_dto()


def test_image_upload_request_should_limit_size():
    file = FileStorage(
        stream=io.BytesIO(b"some big file about 10mb over"),
        filename=f"test.png",
        content_type="multipart/form-data",
    )

    with patch(
        "app.http.requests.v1.user_request.UploadUserProfileImageSchema.get_bytes_size"
    ) as mock_size:
        mock_size.return_value = ImageSizeEnum.ALLOWED_MAX_SIZE_BYTES.value + 1
        with pytest.raises(InvalidRequestException):
            UploadUserProfileImgRequestSchema(file=file).validate_request_and_make_dto()
