from http import HTTPStatus

from pydantic import ValidationError

from app.http.responses import failure_response, success_response
from core.domains.notification.schema.notification_schema import (
    DefaultNotificationResponseSchema,
    GetNotificationResponseSchema,
    GetBadgeResponseSchema,
)
from core.use_case_output import UseCaseSuccessOutput, UseCaseFailureOutput, FailureType


class UpdateNotificationReadPresenter:
    def transform(self, output: UseCaseSuccessOutput | UseCaseFailureOutput):
        if isinstance(output, UseCaseSuccessOutput):
            try:
                schema = DefaultNotificationResponseSchema(result=output.type)
            except ValidationError:
                return failure_response(
                    UseCaseFailureOutput(
                        type="response schema validation error",
                        message=FailureType.INTERNAL_ERROR,
                        code=HTTPStatus.INTERNAL_SERVER_ERROR,
                    ),
                    status_code=HTTPStatus.INTERNAL_SERVER_ERROR,
                )
            result = {
                "data": schema.dict(),
                "meta": output.meta,
            }
            return success_response(result=result)
        elif isinstance(output, UseCaseFailureOutput):
            return failure_response(output=output, status_code=output.code)


class GetNotificationPresenter:
    def transform(self, output: UseCaseSuccessOutput | UseCaseFailureOutput):
        if isinstance(output, UseCaseSuccessOutput):
            try:
                schema = GetNotificationResponseSchema(
                    notifications=output.value.get("data"),
                    cursor=output.value.get("cursor"),
                )
            except ValidationError:
                return failure_response(
                    UseCaseFailureOutput(
                        type="response schema validation error",
                        message=FailureType.INTERNAL_ERROR,
                        code=HTTPStatus.INTERNAL_SERVER_ERROR,
                    ),
                    status_code=HTTPStatus.INTERNAL_SERVER_ERROR,
                )
            result = {
                "data": schema.dict(),
                "meta": output.meta,
            }
            return success_response(result=result)
        elif isinstance(output, UseCaseFailureOutput):
            return failure_response(output=output, status_code=output.code)


class GetBadgePresenter:
    def transform(self, output: UseCaseSuccessOutput | UseCaseFailureOutput):
        if isinstance(output, UseCaseSuccessOutput):
            try:
                schema = GetBadgeResponseSchema(result=output.value)
            except ValidationError:
                return failure_response(
                    UseCaseFailureOutput(
                        type="response schema validation error",
                        message=FailureType.INTERNAL_ERROR,
                        code=HTTPStatus.INTERNAL_SERVER_ERROR,
                    ),
                    status_code=HTTPStatus.INTERNAL_SERVER_ERROR,
                )
            result = {
                "data": schema.dict(),
                "meta": output.meta,
            }
            return success_response(result=result)
        elif isinstance(output, UseCaseFailureOutput):
            return failure_response(output=output, status_code=output.code)
