from flask import request
from flask_jwt_extended import jwt_required

from app.http.requests.v1.admin_request import (
    CreateAdminPostRequest,
    UpdateAdminPostRequest,
    DeleteAdminPostRequest,
    GetAdminPostRequest,
    GetAdminFeedsRequest,
    DeleteAdminCommentRequest,
    GetAdminUserRequest,
    DeleteAdminUserRequest,
    GetAdminCSRequest,
    UpdateAdminCSRequest,
)
from app.http.responses.presenters.v1.admin_presenter import (
    CreateAdminPostPresenter,
    UpdateAdminPostPresenter,
    DeleteAdminPostPresenter,
    GetAdminPostPresenter,
    GetAdminFeedsPresenter,
    DeleteAdminCommentPresenter,
    GetAdminUserPresenter,
    DeleteAdminUserPresenter,
    GetAdminCSPresenter,
    UpdateAdminCSPresenter,
)
from app.http.view import api, auth_required
from app.http.view.authentication import current_user
from core.domains.admin.use_case.v1.admin_use_case import (
    CreateAdminPostUseCase,
    UpdateAdminPostUseCase,
    DeleteAdminPostUseCase,
    GetAdminPostUseCase,
    GetAdminFeedsUseCase,
    DeleteAdminCommentUseCase,
    GetAdminUserUseCase,
    DeleteAdminUserUseCase,
    GetAdminCSUseCase,
    UpdateAdminCSUseCase,
)


@api.route("/v1/admin/post", methods=["POST"])
@jwt_required
@auth_required
def create_admin_post():
    dto = CreateAdminPostRequest(
        **request.form.to_dict(),
        files=request.files.getlist("files"),
        user_id=current_user.id,
    ).validate_request_and_make_dto()

    return CreateAdminPostPresenter().transform(
        CreateAdminPostUseCase().execute(dto=dto)
    )


@api.route("/v1/admin/post/<int:post_id>", methods=["PATCH"])
@jwt_required
@auth_required
def update_admin_post(post_id):
    dto = UpdateAdminPostRequest(
        post_category_id=request.form.get("post_category_id"),
        body=request.form.get("body"),
        title=request.form.get("title"),
        files=request.files.getlist("files"),
        delete_files=request.form.getlist("delete_files"),
        post_id=post_id,
        user_id=current_user.id,
    ).validate_request_and_make_dto()

    return UpdateAdminPostPresenter().transform(
        UpdateAdminPostUseCase().execute(dto=dto)
    )


@api.route("/v1/admin/post/<int:post_id>", methods=["DELETE"])
@jwt_required
@auth_required
def delete_admin_post(post_id):
    dto = DeleteAdminPostRequest(
        post_id=post_id, user_id=current_user.id,
    ).validate_request_and_make_dto()

    return DeleteAdminPostPresenter().transform(
        DeleteAdminPostUseCase().execute(dto=dto)
    )


@api.route("/v1/admin/post", methods=["GET"])
@jwt_required
@auth_required
def get_admin_post():
    dto = GetAdminPostRequest(
        post_id=request.args.get("post_id"), user_id=current_user.id,
    ).validate_request_and_make_dto()

    return GetAdminPostPresenter().transform(GetAdminPostUseCase().execute(dto=dto))


@api.route("/v1/admin/feed", methods=["GET"])
@jwt_required
@auth_required
def get_admin_feeds():
    dto = GetAdminFeedsRequest(
        user_id=current_user.id,
        post_category_id=request.args.get("post_category_id"),
        page_no=request.args.get("page_no"),
        target_user_id=request.args.get("target_user_id"),
    ).validate_request_and_make_dto()

    return GetAdminFeedsPresenter().transform(GetAdminFeedsUseCase().execute(dto=dto))


@api.route("/v1/admin/comment/<int:comment_id>", methods=["DELETE"])
@jwt_required
@auth_required
def delete_admin_comment(comment_id):
    dto = DeleteAdminCommentRequest(
        user_id=current_user.id, comment_id=comment_id
    ).validate_request_and_make_dto()

    return DeleteAdminCommentPresenter().transform(
        DeleteAdminCommentUseCase().execute(dto=dto)
    )


@api.route("/v1/admin/user", methods=["GET"])
@jwt_required
@auth_required
def get_admin_user():
    dto = GetAdminUserRequest(
        target_user_id=request.args.get("target_user_id"),
        nickname=request.args.get("nickname"),
        page_no=request.args.get("page_no"),
        user_id=current_user.id,
    ).validate_request_and_make_dto()

    return GetAdminUserPresenter().transform(GetAdminUserUseCase().execute(dto=dto))


@api.route("/v1/admin/user/<int:target_user_id>", methods=["DELETE"])
@jwt_required
@auth_required
def delete_admin_user(target_user_id):
    dto = DeleteAdminUserRequest(
        target_user_id=target_user_id, user_id=current_user.id,
    ).validate_request_and_make_dto()

    return DeleteAdminUserPresenter().transform(
        DeleteAdminUserUseCase().execute(dto=dto)
    )


@api.route("/v1/admin/cs", methods=["GET"])
@jwt_required
@auth_required
def get_admin_cs_feeds():
    dto = GetAdminCSRequest(
        user_id=current_user.id,
        is_solved=request.args.get("is_solved", default=False, type=bool),
        email=request.args.get("email"),
        page_no=request.args.get("page_no"),
    ).validate_request_and_make_dto()

    return GetAdminCSPresenter().transform(GetAdminCSUseCase().execute(dto=dto))


@api.route("/v1/admin/cs/<int:inquiry_id>", methods=["PATCH"])
@jwt_required
@auth_required
def update_admin_cs_feeds(inquiry_id):
    dto = UpdateAdminCSRequest(
        user_id=current_user.id, inquiry_id=inquiry_id, **request.get_json(),
    ).validate_request_and_make_dto()

    return UpdateAdminCSPresenter().transform(UpdateAdminCSUseCase().execute(dto=dto))
