from sqlalchemy import (
    Column,
    BigInteger,
    DateTime,
    Integer,
    String,
    func,
    Boolean,
    ForeignKey,
    Text,
)
from sqlalchemy.orm import relationship, backref

from app import db
from app.persistence.model.post_model import PostModel
from core.domains.community.entity.community_entity import (
    CommentEntity,
    FeedCommentEntity,
    FeedCommentUserEntity,
)


class CommentModel(db.Model):
    __tablename__ = "comments"

    id = Column(
        BigInteger().with_variant(Integer, "sqlite"),
        primary_key=True,
        nullable=False,
        autoincrement=True,
    )
    post_id = Column(
        BigInteger().with_variant(Integer, "sqlite"),
        ForeignKey(PostModel.id),
        nullable=False,
        index=True,
    )
    user_id = Column(
        BigInteger().with_variant(Integer, "sqlite"), nullable=False, index=True,
    )
    body = Column(Text, nullable=False)
    is_deleted = Column(Boolean, nullable=False, default=False)
    like_count = Column(Integer, nullable=False, default=0)
    last_user_action = Column(String(20), nullable=False)
    last_user_action_at = Column(DateTime(), server_default=func.now(), nullable=False)
    last_admin_action = Column(String(20), nullable=True)
    last_admin_action_at = Column(DateTime(), nullable=True)
    created_at = Column(DateTime(), server_default=func.now(), nullable=False)
    updated_at = Column(
        DateTime(), server_default=func.now(), onupdate=func.now(), nullable=False
    )

    comment_like_statuses = relationship(
        "CommentLikeStatusModel", backref=backref("comments"), uselist=True
    )

    # user relationship
    user = relationship(
        "UserModel",
        backref=backref("comments"),
        uselist=False,
        primaryjoin="CommentModel.user_id == foreign(UserModel.id)",
    )

    def to_entity(self) -> CommentEntity:
        return CommentEntity(
            id=self.id,
            post_id=self.post_id,
            user_id=self.user_id,
            body=self.body,
            is_deleted=self.is_deleted,
            like_count=self.like_count,
            last_user_action=self.last_user_action,
            last_user_action_at=self.last_user_action_at,
            last_admin_action=self.last_admin_action,
            last_admin_action_at=self.last_admin_action_at,
            created_at=self.created_at,
            updated_at=self.updated_at,
            comment_like_statuses=[
                comment_like_status.to_entity()
                for comment_like_status in self.comment_like_statuses
            ]
            if self.comment_like_statuses
            else None,
        )

    def to_feed_comment_entity(
        self, writer_id: int, reader_id: int
    ) -> FeedCommentEntity:
        return FeedCommentEntity(
            id=self.id,
            body=self.body,
            is_liked=True if self.comment_like_statuses else False,
            is_owner=True if self.user_id == writer_id else False,
            is_writer=True if self.user_id == reader_id else False,
            like_count=self.like_count,
            plant_nickname=None,
            user=FeedCommentUserEntity(
                id=self.user.id,
                nickname=self.user.nickname,
                profile_img=self.user.user_profile.path
                if self.user.user_profile
                else None,
            ),
        )
