import UIKit
import KakaoSDKUser
import KakaoSDKAuth
import KakaoSDKCommon

class KakaoLogin: UIViewController {
    
    private let TAG = "KakaoLogin"
    
    // UI 요소
    @IBOutlet weak var logoutButton: UIButton!
    
    // UserDefaults (Android의 SharedPreferences와 동일)
    let userDefaults = UserDefaults.standard
    
    override func viewDidLoad() {
        super.viewDidLoad()
        
        // 전체 화면 설정
        self.navigationController?.isNavigationBarHidden = true
        
        // 로그아웃 버튼 설정
        logoutButton.addTarget(self, action: #selector(onClickLogout), for: .touchUpInside)
        
        // 카카오 로그인 프로세스 시작
        kakaoLoginProcess()
    }
    
    private func kakaoLoginProcess() {
        // 카카오톡 앱을 통한 로그인 가능 여부 확인
        if (AuthApi.isKakaoTalkLoginAvailable()) {
            // 카카오톡 앱으로 로그인
            UserApi.shared.loginWithKakaoTalk { (oauthToken, error) in
                if let error = error {
                    print("\(self.TAG) 로그인 실패: \(error.localizedDescription)")
                } else if let oauthToken = oauthToken {
                    print("\(self.TAG) 로그인 성공: \(oauthToken.accessToken)")
                    self.requestUserInfo()
                }
            }
        } else {
            // 카카오 계정으로 로그인
            UserApi.shared.loginWithKakaoAccount { (oauthToken, error) in
                if let error = error {
                    print("\(self.TAG) 로그인 실패: \(error.localizedDescription)")
                } else if let oauthToken = oauthToken {
                    print("\(self.TAG) 로그인 성공: \(oauthToken.accessToken)")
                    self.requestUserInfo()
                }
            }
        }
    }
    
    private func requestUserInfo() {
        // 사용자 정보 요청
        UserApi.shared.me { (user, error) in
            if let error = error {
                print("\(self.TAG) 사용자 정보 요청 실패: \(error.localizedDescription)")
            } else if let user = user {
                print("\(self.TAG) 사용자 정보 요청 성공: \(user.kakaoAccount?.email ?? "")")
                self.saveUserInfo(user: user)
            }
        }
    }
    
    private func saveUserInfo(user: KakaoSDKUser.User) {
        // 사용자 정보를 UserDefaults에 저장
        let id = String(user.id ?? 0)
        userDefaults.set(id, forKey: "userid")
        userDefaults.set(user.kakaoAccount?.email, forKey: "email")
        userDefaults.set(user.kakaoAccount?.profile?.nickname, forKey: "nickname")
        userDefaults.synchronize()
        
        // WebView에 사용자 정보 전달 (AppDelegate에서 webView 참조 필요)
        DispatchQueue.main.async {
            if let appDelegate = UIApplication.shared.delegate as? AppDelegate,
               let webView = appDelegate.webView {
                let script = "set_user_info('\(id)','\(user.kakaoAccount?.email ?? "")','\(user.kakaoAccount?.profile?.nickname ?? "")')"
                webView.evaluateJavaScript(script, completionHandler: nil)
            }
            
            // 현재 화면 닫기
            self.dismiss(animated: true, completion: nil)
        }
    }
    
    @objc func onClickLogout(_ sender: UIButton) {
        // 로그아웃 처리
        UserApi.shared.logout { (error) in
            if let error = error {
                print("\(self.TAG) 로그아웃 실패: \(error.localizedDescription)")
            } else {
                print("\(self.TAG) 로그아웃 성공")
                
                // UserDefaults 초기화
                self.userDefaults.removeObject(forKey: "userid")
                self.userDefaults.removeObject(forKey: "email")
                self.userDefaults.removeObject(forKey: "nickname")
                self.userDefaults.synchronize()
                
                // 메인 화면으로 이동
                DispatchQueue.main.async {
                    if let appDelegate = UIApplication.shared.delegate as? AppDelegate {
                        let storyboard = UIStoryboard(name: "Main", bundle: nil)
                        let mainViewController = storyboard.instantiateViewController(withIdentifier: "MainViewController")
                        appDelegate.window?.rootViewController = mainViewController
                    }
                }
            }
        }
    }
} 