import UIKit
import AVFoundation
import ZXingObjC

class ReaderViewController: UIViewController, AVCaptureMetadataOutputObjectsDelegate {
    
    var captureSession: AVCaptureSession?
    var previewLayer: AVCaptureVideoPreviewLayer?
    var qrCodeFrameView: UIView?
    
    // ZXing 관련 변수
    private let reader = ZXMultiFormatReader()
    
    override func viewDidLoad() {
        super.viewDidLoad()
        
        // 카메라 권한 확인 및 설정
        setupCamera()
        
        // 스캔 영역 표시를 위한 뷰 설정
        setupQRCodeFrameView()
    }
    
    override func viewWillAppear(_ animated: Bool) {
        super.viewWillAppear(animated)
        
        // 화면이 나타날 때 스캔 시작
        if let captureSession = captureSession, !captureSession.isRunning {
            captureSession.startRunning()
        }
    }
    
    override func viewWillDisappear(_ animated: Bool) {
        super.viewWillDisappear(animated)
        
        // 화면이 사라질 때 스캔 중지
        if let captureSession = captureSession, captureSession.isRunning {
            captureSession.stopRunning()
        }
    }
    
    private func setupCamera() {
        // AVCaptureDevice 설정
        guard let captureDevice = AVCaptureDevice.default(for: .video) else {
            showAlert(title: "카메라 오류", message: "카메라를 사용할 수 없습니다.")
            return
        }
        
        do {
            // 입력 설정
            let input = try AVCaptureDeviceInput(device: captureDevice)
            
            // 세션 설정
            captureSession = AVCaptureSession()
            captureSession?.addInput(input)
            
            // 출력 설정
            let captureMetadataOutput = AVCaptureMetadataOutput()
            captureSession?.addOutput(captureMetadataOutput)
            
            // 델리게이트 설정
            captureMetadataOutput.setMetadataObjectsDelegate(self, queue: DispatchQueue.main)
            captureMetadataOutput.metadataObjectTypes = [
                .qr,
                .code128,
                .code39,
                .code93,
                .ean8,
                .ean13,
                .pdf417,
                .upce,
                .aztec
            ]
            
            // 미리보기 레이어 설정
            previewLayer = AVCaptureVideoPreviewLayer(session: captureSession!)
            previewLayer?.videoGravity = .resizeAspectFill
            previewLayer?.frame = view.layer.bounds
            view.layer.addSublayer(previewLayer!)
            
            // 세션 시작
            captureSession?.startRunning()
            
        } catch {
            showAlert(title: "카메라 오류", message: "카메라를 설정할 수 없습니다: \(error.localizedDescription)")
        }
    }
    
    private func setupQRCodeFrameView() {
        // QR 코드 프레임 뷰 설정
        qrCodeFrameView = UIView()
        qrCodeFrameView?.layer.borderColor = UIColor.green.cgColor
        qrCodeFrameView?.layer.borderWidth = 2
        view.addSubview(qrCodeFrameView!)
        view.bringSubviewToFront(qrCodeFrameView!)
    }
    
    // MARK: - AVCaptureMetadataOutputObjectsDelegate
    
    func metadataOutput(_ output: AVCaptureMetadataOutput, didOutput metadataObjects: [AVMetadataObject], from connection: AVCaptureConnection) {
        // 인식된 메타데이터가 없으면 프레임 뷰 초기화
        if metadataObjects.isEmpty {
            qrCodeFrameView?.frame = CGRect.zero
            return
        }
        
        // 메타데이터 객체가 QR 코드인 경우
        if let metadataObject = metadataObjects.first as? AVMetadataMachineReadableCodeObject {
            // QR 코드 주변에 프레임 표시
            let barCodeObject = previewLayer?.transformedMetadataObject(for: metadataObject)
            qrCodeFrameView?.frame = barCodeObject?.bounds ?? CGRect.zero
            
            // QR 코드 값 처리
            if let stringValue = metadataObject.stringValue {
                handleScannedCode(stringValue)
            }
        }
    }
    
    private func handleScannedCode(_ code: String) {
        // 스캔 일시 중지
        captureSession?.stopRunning()
        
        // 스캔 결과 처리
        let result = code
        
        // 결과 알림
        showAlert(title: "스캔 결과", message: result) { [weak self] in
            // 알림 닫힌 후 스캔 재개
            self?.captureSession?.startRunning()
        }
    }
    
    private func showAlert(title: String, message: String, completion: (() -> Void)? = nil) {
        let alert = UIAlertController(title: title, message: message, preferredStyle: .alert)
        alert.addAction(UIAlertAction(title: "확인", style: .default) { _ in
            completion?()
        })
        present(alert, animated: true)
    }
    
    // MARK: - 화면 회전 처리
    
    override func viewDidLayoutSubviews() {
        super.viewDidLayoutSubviews()
        previewLayer?.frame = view.layer.bounds
    }
} 