# 푸시 알림 시스템 (Push Notification System)

이 시스템은 안드로이드와 iOS 기기에 푸시 알림을 보낼 수 있는 기능을 제공합니다. 개별 푸시와 전체 푸시를 모두 지원하며, 아름다운 웹 인터페이스를 통해 쉽게 사용할 수 있습니다.

## 목차

1. [시스템 개요](#시스템-개요)
2. [기능](#기능)
3. [설치 방법](#설치-방법)
4. [사용 방법](#사용-방법)
   - [웹 인터페이스 사용](#웹-인터페이스-사용)
   - [HTTP API 호출](#http-api-호출)
   - [PHP에서 호출](#php에서-호출)
5. [파일 구조](#파일-구조)
6. [안드로이드 클라이언트 설정](#안드로이드-클라이언트-설정)
7. [iOS 클라이언트 설정](#ios-클라이언트-설정)
8. [문제 해결](#문제-해결)

## 시스템 개요

이 푸시 알림 시스템은 Firebase Cloud Messaging(FCM)을 기반으로 하며, 서버에서 안드로이드와 iOS 기기로 푸시 알림을 보낼 수 있습니다. 시스템은 다음과 같은 구성 요소로 이루어져 있습니다:

- **웹 인터페이스**: 사용자 친화적인 웹 페이지를 통해 푸시 알림을 쉽게 보낼 수 있습니다.
- **HTTP API**: 외부 시스템에서 HTTP 요청을 통해 푸시 알림을 보낼 수 있습니다.
- **PHP 라이브러리**: PHP 애플리케이션에서 직접 푸시 알림 기능을 통합할 수 있습니다.

## 기능

- **개별 푸시**: 특정 사용자(토큰)에게 푸시 알림을 보냅니다.
- **전체 푸시**: 모든 사용자에게 푸시 알림을 보냅니다.
- **안드로이드 및 iOS 지원**: 두 플랫폼 모두에 최적화된 푸시 알림을 보낼 수 있습니다.
- **이미지 지원**: 푸시 알림에 이미지를 포함할 수 있습니다.
- **URL 연결**: 푸시 알림을 탭하면 특정 URL로 이동할 수 있습니다.
- **실시간 미리보기**: 웹 인터페이스에서 푸시 알림의 모양을 미리 볼 수 있습니다.

## 설치 방법

1. 모든 파일을 웹 서버의 디렉토리에 업로드합니다.
2. Firebase 서비스 계정 키 파일(`mars-38372-firebase-adminsdk-60l5a-23f211e854.json`)이 올바른 위치에 있는지 확인합니다.
3. 필요한 경우 `send_push.php` 및 `push_api.php` 파일에서 Firebase 프로젝트 ID와 서비스 계정 경로를 수정합니다.

## 사용 방법

### 웹 인터페이스 사용

1. 웹 브라우저에서 `push_notification.php` 파일에 접속합니다.
2. 개별 푸시 또는 전체 푸시 탭을 선택합니다.
3. 필요한 정보를 입력합니다:
   - 기기 유형(안드로이드 또는 iOS)
   - 사용자 토큰(개별 푸시의 경우)
   - 알림 제목
   - 알림 내용
   - URL(선택사항)
   - 이미지 URL(선택사항)
4. "푸시 알림 보내기" 버튼을 클릭합니다.

### HTTP API 호출

크롬 또는 다른 브라우저에서 URL을 직접 입력하여 푸시 알림을 보낼 수 있습니다:

```
http://your-domain.com/push_api.php?app=android&user_token=DEVICE_TOKEN&title=알림제목&memo=알림내용&url=https://example.com&image_url=https://example.com/image.jpg
```

#### 필수 파라미터:
- `app`: 기기 유형 (android 또는 ios)
- `user_token`: 사용자 FCM 토큰
- `title`: 알림 제목
- `memo`: 알림 내용

#### 선택적 파라미터:
- `url`: 알림 클릭 시 이동할 URL
- `image_url`: 알림에 표시할 이미지 URL

### PHP에서 호출

PHP 코드에서 직접 푸시 알림 함수를 호출할 수 있습니다:

```php
<?php
// 푸시 알림 함수 호출
$app = 'android'; // 또는 'ios'
$user_token = 'DEVICE_TOKEN';
$title = '알림 제목';
$memo = '알림 내용';
$url = 'https://example.com';
$image_url = 'https://example.com/image.jpg';

// 함수 호출
$result = send_push_v1_jwt($app, $user_token, $title, $memo, $url, $image_url);

if ($result) {
    echo "푸시 알림이 성공적으로 전송되었습니다.";
} else {
    echo "푸시 알림 전송에 실패했습니다.";
}
?>
```

## 파일 구조

- `push_notification.php`: 웹 인터페이스를 제공하는 HTML 파일
- `send_push.php`: 웹 인터페이스에서 호출되는 푸시 알림 처리 PHP 파일
- `push_api.php`: HTTP API를 통해 푸시 알림을 보낼 수 있는 PHP 파일
- `mars-38372-firebase-adminsdk-60l5a-23f211e854.json`: Firebase 서비스 계정 키 파일

## 안드로이드 클라이언트 설정

안드로이드 앱에서 푸시 알림을 수신하려면 다음 단계를 따르세요:

1. Firebase 프로젝트에 안드로이드 앱을 추가합니다.
2. `google-services.json` 파일을 앱 프로젝트에 추가합니다.
3. Firebase Cloud Messaging 종속성을 추가합니다.
4. 다음과 같은 서비스를 구현합니다:

```java
public class MyFirebaseMessagingService extends FirebaseMessagingService {
    @Override
    public void onMessageReceived(RemoteMessage remoteMessage) {
        // 푸시 알림 처리
        if (remoteMessage.getData().size() > 0) {
            String title = remoteMessage.getData().get("title");
            String message = remoteMessage.getData().get("message");
            String url = remoteMessage.getData().get("url");
            String file_url = remoteMessage.getData().get("file_url");
            
            // 알림 표시
            sendNotification(title, message, url, file_url);
        }
    }
    
    @Override
    public void onNewToken(String token) {
        // 새 토큰을 서버에 등록
        sendRegistrationToServer(token);
    }
    
    private void sendNotification(String title, String message, String url, String imageUrl) {
        // 알림 생성 및 표시 코드
    }
}
```

## iOS 클라이언트 설정

iOS 앱에서 푸시 알림을 수신하려면 다음 단계를 따르세요:

1. Firebase 프로젝트에 iOS 앱을 추가합니다.
2. `GoogleService-Info.plist` 파일을 앱 프로젝트에 추가합니다.
3. Firebase Cloud Messaging 종속성을 추가합니다.
4. 앱 델리게이트에서 다음 코드를 구현합니다:

```swift
import UIKit
import Firebase
import FirebaseMessaging
import UserNotifications

@UIApplicationMain
class AppDelegate: UIResponder, UIApplicationDelegate, MessagingDelegate, UNUserNotificationCenterDelegate {
    func application(_ application: UIApplication, didFinishLaunchingWithOptions launchOptions: [UIApplication.LaunchOptionsKey: Any]?) -> Bool {
        FirebaseApp.configure()
        
        // 푸시 알림 권한 요청
        UNUserNotificationCenter.current().delegate = self
        let authOptions: UNAuthorizationOptions = [.alert, .badge, .sound]
        UNUserNotificationCenter.current().requestAuthorization(options: authOptions) { _, _ in }
        
        application.registerForRemoteNotifications()
        
        Messaging.messaging().delegate = self
        
        return true
    }
    
    func messaging(_ messaging: Messaging, didReceiveRegistrationToken fcmToken: String?) {
        // 토큰을 서버에 등록
        if let token = fcmToken {
            sendRegistrationToServer(token)
        }
    }
    
    func userNotificationCenter(_ center: UNUserNotificationCenter, willPresent notification: UNNotification, withCompletionHandler completionHandler: @escaping (UNNotificationPresentationOptions) -> Void) {
        completionHandler([.alert, .badge, .sound])
    }
    
    func userNotificationCenter(_ center: UNUserNotificationCenter, didReceive response: UNNotificationResponse, withCompletionHandler completionHandler: @escaping () -> Void) {
        // 알림 탭 처리
        let userInfo = response.notification.request.content.userInfo
        
        if let urlString = userInfo["url"] as? String, let url = URL(string: urlString) {
            UIApplication.shared.open(url)
        }
        
        completionHandler()
    }
}
```

## 문제 해결

### 푸시 알림이 전송되지 않는 경우:

1. Firebase 서비스 계정 키 파일이 올바른 위치에 있는지 확인합니다.
2. 사용자 토큰이 유효한지 확인합니다.
3. 서버 오류 로그를 확인합니다.
4. Firebase 콘솔에서 프로젝트 설정이 올바른지 확인합니다.

### 안드로이드에서 알림이 표시되지 않는 경우:

1. 앱이 백그라운드에 있는지 확인합니다.
2. 알림 채널 설정을 확인합니다.
3. 기기의 알림 설정을 확인합니다.

### iOS에서 알림이 표시되지 않는 경우:

1. 앱이 알림 권한을 가지고 있는지 확인합니다.
2. 프로비저닝 프로파일에 푸시 알림 기능이 활성화되어 있는지 확인합니다.
3. APNs 인증서가 Firebase 콘솔에 올바르게 업로드되었는지 확인합니다.



테스트

https://waterreg.andpia.com/new_http_v1/push_notification.php




// Fetch API 사용
fetch('http://waterreg.andpia.com/new_http_v1/push_api.php?app=android&user_token=FCM_TOKEN&title=JS테스트&memo=자바스크립트테스트&url=http://waterreg.andpia.com')
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            console.log('푸시 알림 전송 성공!');
        } else {
            console.error('푸시 알림 전송 실패:', data.message);
        }
    })
    .catch(error => console.error('오류 발생:', error));

// jQuery AJAX 사용
$.ajax({
    url: 'http://waterreg.andpia.com/new_http_v1/push_api.php',
    type: 