<?php
/**
 * push.php
 * 푸시 알림 테스트를 위한 독립 실행형 PHP 파일
 * 
 * 개별 푸시 알림과 그룹 푸시 알림을 테스트할 수 있는 간단한 폼을 제공합니다.
 * 데이터베이스 연결 없이 작동합니다.
 */

// 오류 표시 설정
ini_set('display_errors', 1);
error_reporting(E_ALL);

// 푸시 알림 전송 함수
function push_send($tokens, $message, $title, $send_type = 'test', $image_url = "", $click_action = "") {
    // Firebase 프로젝트 ID
    $PROJECT_ID = 'compcareadmin';
    
    // 결과 배열 초기화
    $ref = array();
    $ref['result'] = 'error';
    $ref['msg'] = '';
    $ref['push_result'] = array();
    
    // FCM 토큰이 없는 경우 처리
    if (!$tokens) {
        $ref['msg'] = 'FCM 토큰이 없습니다.';
        return $ref;
    }
    
    // 토큰을 배열로 변환
    if (!is_array($tokens)) {
        $tokens = array($tokens);
    }
    
    // 중복 토큰 제거 및 빈 토큰 필터링
    $tokens = array_unique($tokens);
    $tokens = array_filter($tokens);
    
    // 토큰이 없는 경우 처리
    if (count($tokens) == 0) {
        $ref['msg'] = '유효한 FCM 토큰이 없습니다.';
        return $ref;
    }
    
    // OAuth 토큰 가져오기
    $oauth_token = get_oauth_token();
    
    // 헤더 설정
    $headers = array();
    if ($oauth_token) {
        // OAuth 인증 사용
        $headers[] = 'Authorization: Bearer ' . $oauth_token;
        $headers[] = 'Content-Type: application/json';
    } else {
        // 레거시 서버 키 인증 사용 (OAuth 실패 시 폴백)
        $headers[] = 'Authorization: key=AAAA-Wd_Ztg:APA91bGKxRhsLRNWJJbBjZQtm4nLxuBdMepv-xFBFXKYJYXwbBJLXYlVBNOJyFmYWJxX3EZQQtZY-_LrJQWRgNRrB-zEkJgxqCVmQVNebKWxNZJzGHYbIkxlXLQoNKqzDwHbz9SXnOZA';
        $headers[] = 'Content-Type: application/json';
    }
    
    // 성공 및 실패 카운트 초기화
    $success_count = 0;
    $fail_count = 0;
    $results = array();
    
    // 각 토큰에 대해 메시지 전송
    foreach ($tokens as $token) {
        // 메시지 데이터 구성
        $message_data = array(
            'message' => array(
                'token' => $token,
                'notification' => array(
                    'title' => $title,
                    'body' => $message
                ),
                'data' => array(
                    'title' => $title,
                    'message' => $message,
                    'send_type' => $send_type
                )
            )
        );
        
        // 이미지 URL이 있는 경우 추가
        if ($image_url) {
            $message_data['message']['notification']['image'] = $image_url;
            $message_data['message']['data']['image_url'] = $image_url;
        }
        
        // 클릭 액션 URL이 있는 경우 추가
        if ($click_action) {
            $message_data['message']['data']['click_action'] = $click_action;
        }
        
        // Android 특정 설정
        $message_data['message']['android'] = array(
            'priority' => 'high',
            'notification' => array(
                'sound' => 'default'
            )
        );
        
        // iOS 특정 설정
        $message_data['message']['apns'] = array(
            'headers' => array(
                'apns-priority' => '10'
            ),
            'payload' => array(
                'aps' => array(
                    'sound' => 'default',
                    'badge' => 1
                )
            )
        );
        
        // cURL 초기화
        $ch = curl_init();
        
        // cURL 옵션 설정
        curl_setopt($ch, CURLOPT_URL, "https://fcm.googleapis.com/v1/projects/{$PROJECT_ID}/messages:send");
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($message_data));
        
        // 요청 실행
        $result = curl_exec($ch);
        $http_status = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        
        // cURL 종료
        curl_close($ch);
        
        // 결과 처리
        $result_data = json_decode($result, true);
        
        if ($http_status == 200) {
            $success_count++;
            $results[] = array(
                'token' => $token,
                'success' => true,
                'message_id' => isset($result_data['name']) ? $result_data['name'] : ''
            );
        } else {
            $fail_count++;
            $error_message = isset($result_data['error']['message']) ? $result_data['error']['message'] : '알 수 없는 오류';
            $results[] = array(
                'token' => $token,
                'success' => false,
                'error' => $error_message
            );
        }
    }
    
    // 최종 결과 설정
    $ref['push_result'] = $results;
    
    if ($success_count > 0) {
        $ref['result'] = 'success';
        $ref['msg'] = "총 {$success_count}개의 알림이 성공적으로 전송되었습니다.";
        if ($fail_count > 0) {
            $ref['msg'] .= " ({$fail_count}개 실패)";
        }
    } else {
        $ref['msg'] = "알림 전송에 실패했습니다.";
    }
    
    return $ref;
}

// OAuth 토큰 가져오기 함수
function get_oauth_token() {
    $service_account_file = __DIR__ . '/compcareadmin-firebase-adminsdk-6ebhn-7c69aab0ba.json';
    
    if (!file_exists($service_account_file)) {
        return false;
    }
    
    $service_account = json_decode(file_get_contents($service_account_file), true);
    
    if (!isset($service_account['private_key']) || !isset($service_account['client_email'])) {
        return false;
    }
    
    // JWT 헤더 생성
    $header = [
        'alg' => 'RS256',
        'typ' => 'JWT'
    ];
    
    // 현재 시간과 만료 시간 설정 (1시간)
    $now = time();
    $exp = $now + 3600;
    
    // JWT 클레임 생성
    $claims = [
        'iss' => $service_account['client_email'],
        'scope' => 'https://www.googleapis.com/auth/firebase.messaging',
        'aud' => 'https://oauth2.googleapis.com/token',
        'exp' => $exp,
        'iat' => $now
    ];
    
    // JWT 인코딩
    $base64_header = str_replace(['+', '/', '='], ['-', '_', ''], base64_encode(json_encode($header)));
    $base64_claims = str_replace(['+', '/', '='], ['-', '_', ''], base64_encode(json_encode($claims)));
    
    $jwt_unsigned = $base64_header . '.' . $base64_claims;
    
    // JWT 서명
    $private_key = $service_account['private_key'];
    openssl_sign(
        $jwt_unsigned,
        $signature,
        $private_key,
        'SHA256'
    );
    
    $base64_signature = str_replace(['+', '/', '='], ['-', '_', ''], base64_encode($signature));
    
    // 완성된 JWT
    $jwt = $jwt_unsigned . '.' . $base64_signature;
    
    // OAuth 토큰 요청
    $ch = curl_init('https://oauth2.googleapis.com/token');
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query([
        'grant_type' => 'urn:ietf:params:oauth:grant-type:jwt-bearer',
        'assertion' => $jwt
    ]));
    
    $response = curl_exec($ch);
    curl_close($ch);
    
    $token_data = json_decode($response, true);
    
    // 액세스 토큰 반환
    if (isset($token_data['access_token'])) {
        return $token_data['access_token'];
    }
    
    return false;
}

// 폼 제출 처리
$result = null;
$message = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $push_type = isset($_POST['push_type']) ? $_POST['push_type'] : '';
    $title = isset($_POST['title']) ? $_POST['title'] : '';
    $message_content = isset($_POST['message']) ? $_POST['message'] : '';
    $image_url = isset($_POST['image_url']) ? $_POST['image_url'] : '';
    $click_action = isset($_POST['click_action']) ? $_POST['click_action'] : '';
    $token = isset($_POST['token']) ? $_POST['token'] : '';
    
    if (empty($token)) {
        $message = '오류: FCM 토큰을 입력해주세요.';
    } else if (empty($title)) {
        $message = '오류: 제목을 입력해주세요.';
    } else if (empty($message_content)) {
        $message = '오류: 내용을 입력해주세요.';
    } else {
        // 토큰을 배열로 변환 (쉼표로 구분된 여러 토큰 지원)
        $token_array = explode(',', $token);
        
        // 푸시 알림 전송
        $result = push_send($token_array, $message_content, $title, $push_type, $image_url, $click_action);
        
        if ($result['result'] === 'success') {
            $message = '성공: ' . $result['msg'];
        } else {
            $message = '오류: ' . $result['msg'];
        }
    }
}
?>
<!DOCTYPE html>
<html>
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>푸시 알림 테스트</title>
    <link rel="stylesheet" href="https://stackpath.bootstrapcdn.com/bootstrap/4.5.2/css/bootstrap.min.css">
    <style>
        body { padding: 20px; }
        .container { max-width: 800px; }
        .tab-content { padding: 20px; border: 1px solid #dee2e6; border-top: none; }
        .result { margin-top: 20px; }
        pre { background: #f8f9fa; padding: 15px; border-radius: 5px; }
        .alert { margin-top: 20px; }
    </style>
</head>
<body>
    <div class="container">
        <h1 class="mb-4">푸시 알림 테스트</h1>
        
        <?php if (!empty($message)): ?>
            <div class="alert <?php echo strpos($message, '성공') !== false ? 'alert-success' : 'alert-danger'; ?>">
                <?php echo $message; ?>
            </div>
        <?php endif; ?>
        
        <ul class="nav nav-tabs" id="pushTabs" role="tablist">
            <li class="nav-item">
                <a class="nav-link active" id="individual-tab" data-toggle="tab" href="#individual" role="tab">개별 알림</a>
            </li>
            <li class="nav-item">
                <a class="nav-link" id="group-tab" data-toggle="tab" href="#group" role="tab">그룹 알림</a>
            </li>
        </ul>
        
        <div class="tab-content">
            <!-- 개별 알림 탭 -->
            <div class="tab-pane fade show active" id="individual" role="tabpanel">
                <form method="post" action="">
                    <input type="hidden" name="push_type" value="individual">
                    
                    <div class="form-group">
                        <label for="token">FCM 토큰</label>
                        <textarea class="form-control" id="token" name="token" rows="3" required></textarea>
                        <small class="form-text text-muted">테스트할 FCM 토큰을 입력하세요.</small>
                    </div>
                    
                    <div class="form-group">
                        <label for="title">제목</label>
                        <input type="text" class="form-control" id="title" name="title" required>
                    </div>
                    
                    <div class="form-group">
                        <label for="message">내용</label>
                        <textarea class="form-control" id="message" name="message" rows="3" required></textarea>
                    </div>
                    
                    <div class="form-group">
                        <label for="image_url">이미지 URL (선택사항)</label>
                        <input type="url" class="form-control" id="image_url" name="image_url">
                    </div>
                    
                    <div class="form-group">
                        <label for="click_action">클릭 액션 URL (선택사항)</label>
                        <input type="url" class="form-control" id="click_action" name="click_action">
                    </div>
                    
                    <button type="submit" class="btn btn-primary">알림 전송</button>
                </form>
            </div>
            
            <!-- 그룹 알림 탭 -->
            <div class="tab-pane fade" id="group" role="tabpanel">
                <form method="post" action="">
                    <input type="hidden" name="push_type" value="group">
                    
                    <div class="form-group">
                        <label for="token_group">FCM 토큰 목록</label>
                        <textarea class="form-control" id="token_group" name="token" rows="5" required></textarea>
                        <small class="form-text text-muted">여러 FCM 토큰을 쉼표(,)로 구분하여 입력하세요.</small>
                    </div>
                    
                    <div class="form-group">
                        <label for="title_group">제목</label>
                        <input type="text" class="form-control" id="title_group" name="title" required>
                    </div>
                    
                    <div class="form-group">
                        <label for="message_group">내용</label>
                        <textarea class="form-control" id="message_group" name="message" rows="3" required></textarea>
                    </div>
                    
                    <div class="form-group">
                        <label for="image_url_group">이미지 URL (선택사항)</label>
                        <input type="url" class="form-control" id="image_url_group" name="image_url">
                    </div>
                    
                    <div class="form-group">
                        <label for="click_action_group">클릭 액션 URL (선택사항)</label>
                        <input type="url" class="form-control" id="click_action_group" name="click_action">
                    </div>
                    
                    <button type="submit" class="btn btn-primary">알림 전송</button>
                </form>
            </div>
        </div>
        
        <?php if ($result): ?>
            <div class="result">
                <h3>전송 결과</h3>
                <pre><?php print_r($result); ?></pre>
            </div>
        <?php endif; ?>
    </div>
    
    <script src="https://code.jquery.com/jquery-3.5.1.slim.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/popper.js@1.16.1/dist/umd/popper.min.js"></script>
    <script src="https://stackpath.bootstrapcdn.com/bootstrap/4.5.2/js/bootstrap.min.js"></script>
</body>
</html>
