/**
 * Push_controller.php
 * 푸시 알림 관리 컨트롤러
 * 
 * 개별 및 전체 푸시 알림을 보내는 기능을 제공합니다.
 * HTTP v1 API를 사용하여 Firebase Cloud Messaging으로 알림을 전송합니다.
 * 
 * @author Claude
 * @version 1.0
 * @date 2023-03-05
 */
class Push_controller extends MY_Controller {
    
    /**
     * 생성자
     * 필요한 모델, 헬퍼, 라이브러리를 로드합니다.
     */
    function __construct() {
        parent::__construct();
        
        // 필요한 모델 로드
        $this->load->model('clib_model');
        
        // 폼 검증 라이브러리 로드
        $this->load->library('form_validation');
    }
    
    /**
     * 개별 푸시 알림 전송 페이지
     * 
     * 개별 사용자에게 푸시 알림을 보내는 폼을 표시합니다.
     * POST 요청 시 입력값을 검증하고 알림을 전송합니다.
     */
    public function individual() {
        // 폼 검증 규칙 설정
        $this->form_validation->set_rules('token', 'FCM 토큰', 'required');
        $this->form_validation->set_rules('title', '제목', 'required');
        $this->form_validation->set_rules('message', '내용', 'required');
        
        // 폼 검증 실행
        if ($this->form_validation->run() == FALSE) {
            // 검증 실패 시 폼 다시 표시
            $this->load->view('push_admin/views/individual_view');
        } else {
            // 입력값 가져오기
            $token = $this->input->post('token');
            $title = $this->input->post('title');
            $message = $this->input->post('message');
            $image_url = $this->input->post('image_url');
            $click_action = $this->input->post('click_action');
            
            // 토큰을 배열로 변환 (쉼표로 구분된 여러 토큰 지원)
            $token_array = explode(',', $token);
            
            // 푸시 알림 전송
            $result = $this->clib_model->push_send($token_array, $message, $title, 'individual', $image_url, $click_action);
            
            // 결과에 따른 메시지 설정
            if ($result['result'] == 'success') {
                $this->session->set_flashdata('message', '푸시 알림이 성공적으로 전송되었습니다.');
            } else {
                $this->session->set_flashdata('error', '푸시 알림 전송 실패: ' . $result['msg']);
            }
            
            // 페이지 리다이렉트
            redirect('push_admin/push_controller/individual');
        }
    }
    
    /**
     * 전체 푸시 알림 전송 페이지
     * 
     * 사용자 유형(기사, 공장, 전체)에 따라 여러 사용자에게 
     * 푸시 알림을 보내는 폼을 표시합니다.
     * POST 요청 시 입력값을 검증하고 알림을 전송합니다.
     */
    public function broadcast() {
        // 폼 검증 규칙 설정
        $this->form_validation->set_rules('title', '제목', 'required');
        $this->form_validation->set_rules('message', '내용', 'required');
        $this->form_validation->set_rules('user_type', '사용자 유형', 'required');
        
        // 폼 검증 실행
        if ($this->form_validation->run() == FALSE) {
            // 검증 실패 시 폼 다시 표시
            $this->load->view('push_admin/views/broadcast_view');
        } else {
            // 입력값 가져오기
            $title = $this->input->post('title');
            $message = $this->input->post('message');
            $user_type = $this->input->post('user_type');
            $image_url = $this->input->post('image_url');
            $click_action = $this->input->post('click_action');
            
            // 사용자 유형에 따른 쿼리 조건 설정
            $where = "";
            if ($user_type == 'engineer') {
                $where = "mb_type = 'E' AND mem_fcm_key IS NOT NULL AND mem_fcm_key != ''";
            } else if ($user_type == 'company') {
                $where = "mb_type = 'C' AND mem_fcm_key IS NOT NULL AND mem_fcm_key != ''";
            } else {
                $where = "mem_fcm_key IS NOT NULL AND mem_fcm_key != ''";
            }
            
            // 사용자 토큰 가져오기
            $users = $this->clib_model->user_fetch($where);
            $tokens = array();
            
            // 토큰 배열 생성
            foreach ($users as $user) {
                if (!empty($user['mem_fcm_key'])) {
                    $tokens[] = $user['mem_fcm_key'];
                }
            }
            
            // 토큰이 있는 경우에만 푸시 알림 전송
            if (count($tokens) > 0) {
                $result = $this->clib_model->push_send($tokens, $message, $title, 'broadcast', $image_url, $click_action);
                
                // 결과에 따른 메시지 설정
                if ($result['result'] == 'success') {
                    $this->session->set_flashdata('message', count($tokens) . '명의 사용자에게 푸시 알림이 성공적으로 전송되었습니다.');
                } else {
                    $this->session->set_flashdata('error', '푸시 알림 전송 실패: ' . $result['msg']);
                }
            } else {
                $this->session->set_flashdata('error', '전송할 사용자가 없습니다.');
            }
            
            // 페이지 리다이렉트
            redirect('push_admin/push_controller/broadcast');
        }
    }
} 