/**
 * Push_test.php
 * 푸시 알림 테스트 컨트롤러
 * 
 * 푸시 알림 기능을 테스트하기 위한 다양한 메소드를 제공합니다.
 * 개별 알림, 전체 알림, 토큰 테스트 등의 기능이 포함되어 있습니다.
 * 
 * @author Claude
 * @version 1.0
 * @date 2023-03-05
 */
class Push_test extends CI_Controller {
    
    /**
     * 생성자
     * 필요한 모델을 로드합니다.
     */
    function __construct() {
        parent::__construct();
        
        // clib_model 로드
        $this->load->model('clib_model');
    }
    
    /**
     * 개별 푸시 알림 테스트
     * 
     * 하드코딩된 테스트 토큰을 사용하여 개별 푸시 알림을 전송합니다.
     * 테스트 목적으로만 사용됩니다.
     */
    public function individual_test() {
        // 테스트 토큰 (실제 테스트 시 유효한 토큰으로 변경 필요)
        $test_token = "테스트_토큰을_여기에_입력하세요";
        
        // 알림 내용 설정
        $title = "개별 알림 테스트";
        $message = "이것은 개별 푸시 알림 테스트입니다. " . date('Y-m-d H:i:s');
        $image_url = "https://example.com/image.jpg"; // 선택사항
        $click_action = "https://example.com"; // 선택사항
        
        // 푸시 알림 전송
        $result = $this->clib_model->push_send($test_token, $message, $title, 'test', $image_url, $click_action);
        
        // 결과 출력
        echo "<h1>개별 푸시 알림 테스트</h1>";
        echo "<pre>";
        print_r($result);
        echo "</pre>";
    }
    
    /**
     * 전체 푸시 알림 테스트
     * 
     * 사용자 유형(기사, 공장, 전체)에 따라 여러 사용자에게 
     * 테스트 푸시 알림을 전송합니다.
     * 
     * @param string $user_type 사용자 유형 (engineer, company, all)
     */
    public function broadcast_test($user_type = 'all') {
        // 사용자 유형에 따른 쿼리 조건 설정
        $where = "";
        if ($user_type == 'engineer') {
            $where = "mb_type = 'E' AND mem_fcm_key IS NOT NULL AND mem_fcm_key != ''";
            $type_name = "기사";
        } else if ($user_type == 'company') {
            $where = "mb_type = 'C' AND mem_fcm_key IS NOT NULL AND mem_fcm_key != ''";
            $type_name = "공장";
        } else {
            $where = "mem_fcm_key IS NOT NULL AND mem_fcm_key != ''";
            $type_name = "전체";
        }
        
        // 알림 내용 설정
        $title = "{$type_name} 알림 테스트";
        $message = "이것은 {$type_name} 푸시 알림 테스트입니다. " . date('Y-m-d H:i:s');
        $image_url = "https://example.com/image.jpg"; // 선택사항
        $click_action = "https://example.com"; // 선택사항
        
        // 사용자 토큰 가져오기
        $users = $this->clib_model->user_fetch($where);
        $tokens = array();
        
        // 토큰 배열 생성
        foreach ($users as $user) {
            if (!empty($user['mem_fcm_key'])) {
                $tokens[] = $user['mem_fcm_key'];
            }
        }
        
        // 결과 출력 시작
        echo "<h1>{$type_name} 푸시 알림 테스트</h1>";
        echo "<p>대상 사용자 수: " . count($tokens) . "명</p>";
        
        // 토큰이 있는 경우에만 푸시 알림 전송
        if (count($tokens) > 0) {
            $result = $this->clib_model->push_send($tokens, $message, $title, 'test', $image_url, $click_action);
            
            echo "<pre>";
            print_r($result);
            echo "</pre>";
        } else {
            echo "<p>푸시 알림을 받을 수 있는 사용자가 없습니다.</p>";
        }
    }
    
    /**
     * 토큰 테스트 폼
     * 
     * 사용자가 직접 FCM 토큰과 알림 내용을 입력하여
     * 푸시 알림을 테스트할 수 있는 폼을 제공합니다.
     */
    public function token_test() {
        // 폼 HTML 출력
        echo '<!DOCTYPE html>
        <html>
        <head>
            <meta charset="UTF-8">
            <meta name="viewport" content="width=device-width, initial-scale=1.0">
            <title>FCM 토큰 테스트</title>
            <link rel="stylesheet" href="https://stackpath.bootstrapcdn.com/bootstrap/4.5.2/css/bootstrap.min.css">
            <style>
                body { padding: 20px; }
                .container { max-width: 800px; }
            </style>
        </head>
        <body>
            <div class="container">
                <h1 class="mb-4">FCM 토큰 테스트</h1>
                
                <form action="' . site_url('push_admin/push_test/send_token_test') . '" method="post">
                    <div class="form-group">
                        <label for="token">FCM 토큰</label>
                        <textarea class="form-control" id="token" name="token" rows="3" required></textarea>
                        <small class="form-text text-muted">테스트할 FCM 토큰을 입력하세요. 여러 토큰은 쉼표로 구분하세요.</small>
                    </div>
                    
                    <div class="form-group">
                        <label for="title">제목</label>
                        <input type="text" class="form-control" id="title" name="title" required>
                    </div>
                    
                    <div class="form-group">
                        <label for="message">내용</label>
                        <textarea class="form-control" id="message" name="message" rows="3" required></textarea>
                    </div>
                    
                    <div class="form-group">
                        <label for="image_url">이미지 URL (선택사항)</label>
                        <input type="url" class="form-control" id="image_url" name="image_url">
                    </div>
                    
                    <div class="form-group">
                        <label for="click_action">클릭 액션 URL (선택사항)</label>
                        <input type="url" class="form-control" id="click_action" name="click_action">
                    </div>
                    
                    <button type="submit" class="btn btn-primary">알림 전송</button>
                </form>
            </div>
        </body>
        </html>';
    }
    
    /**
     * 토큰 테스트 실행
     * 
     * 사용자가 입력한 FCM 토큰과 알림 내용으로
     * 푸시 알림을 전송하고 결과를 표시합니다.
     */
    public function send_token_test() {
        // POST 데이터 가져오기
        $token = $this->input->post('token');
        $title = $this->input->post('title');
        $message = $this->input->post('message');
        $image_url = $this->input->post('image_url');
        $click_action = $this->input->post('click_action');
        
        // 토큰을 배열로 변환 (쉼표로 구분된 여러 토큰 지원)
        $token_array = explode(',', $token);
        
        // 결과 출력 시작
        echo '<!DOCTYPE html>
        <html>
        <head>
            <meta charset="UTF-8">
            <meta name="viewport" content="width=device-width, initial-scale=1.0">
            <title>FCM 토큰 테스트 결과</title>
            <link rel="stylesheet" href="https://stackpath.bootstrapcdn.com/bootstrap/4.5.2/css/bootstrap.min.css">
            <style>
                body { padding: 20px; }
                .container { max-width: 800px; }
                pre { background: #f8f9fa; padding: 15px; border-radius: 5px; }
            </style>
        </head>
        <body>
            <div class="container">
                <h1 class="mb-4">FCM 토큰 테스트 결과</h1>';
        
        // 푸시 알림 전송
        $result = $this->clib_model->push_send($token_array, $message, $title, 'test', $image_url, $click_action);
        
        // 결과 출력
        echo '<pre>';
        print_r($result);
        echo '</pre>';
        
        echo '<a href="' . site_url('push_admin/push_test/token_test') . '" class="btn btn-primary">다시 테스트</a>';
        echo '</div></body></html>';
    }
} 